/* THIS PROGRAM IS PROVIDED "AS IS". TI MAKES NO WARRANTIES OR REPRESENTATIONS,
 * EITHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING ANY IMPLIED WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, LACK OF VIRUSES, ACCURACY
 * OR COMPLETENESS OF RESPONSES, RESULTS AND LACK OF NEGLIGENCE. TI DISCLAIMS
 * ANY WARRANTY OF TITLE, QUIET ENJOYMENT, QUIET POSSESSION, AND NON-INFRINGEMENT
 * OF ANY THIRD PARTY INTELLECTUAL PROPERTY RIGHTS WITH REGARD TO THE PROGRAM OR
 * YOUR USE OF THE PROGRAM.
 * IN NO EVENT SHALL TI BE LIABLE FOR ANY SPECIAL, INCIDENTAL, CONSEQUENTIAL OR
 * INDIRECT DAMAGES, HOWEVER CAUSED, ON ANY THEORY OF LIABILITY AND WHETHER OR
 * NOT TI HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGES, ARISING IN ANY WAY
 * OUT OF THIS AGREEMENT, THE PROGRAM, OR YOUR USE OF THE PROGRAM.  EXCLUDED
 * DAMAGES INCLUDE, BUT ARE NOT LIMITED TO, COST OF REMOVAL OR REINSTALLATION,
 * COMPUTER TIME, LABOR COSTS, LOSS OF GOODWILL, LOSS OF PROFITS, LOSS OF
 * SAVINGS, OR LOSS OF USE OR INTERRUPTION OF BUSINESS. IN NO EVENT WILL TI'S
 * AGGREGATE LIABILITY UNDER THIS AGREEMENT OR ARISING OUT OF YOUR USE OF THE
 * PROGRAM EXCEED FIVE HUNDRED DOLLARS (U.S.$500).
 * Unless otherwise stated, the Program written and copyrighted by Texas
 * Instruments is distributed as "freeware".  You may, only under TI's copyright
 * in the Program, use and modify the Program without any charge or restriction.
 * You may distribute to third parties, provided that you transfer a copy of this
 * license to the third party and the third party agrees to these terms by its
 * first use of the Program. You must reproduce the copyright notice and any
 * other legend of ownership on each copy or partial copy, of the Program.
 * You acknowledge and agree that the Program contains copyrighted material,
 * trade secrets and other TI proprietary information and is protected by
 * copyright laws, international copyright treaties, and trade secret laws, as
 * well as other intellectual property laws.  To protect TI's rights in the
 * Program, you agree not to decompile, reverse engineer, disassemble or
 * otherwise translate any object code versions of the Program to a
 * human-readable form.  You agree that in no event will you alter, remove or
 * destroy any copyright notice included in the Program.  TI reserves all rights
 * not specifically granted under this license. Except as specifically provided
 * herein, nothing in this agreement shall be construed as conferring by
 * implication, estoppel, or otherwise, upon you, any license or other right
 * under any TI patents, copyrights or trade secrets.
 * You may not use the Program in non-TI devices. */


//******************************************************************************
//
//  adc.c
//  Functions for using the integrated AD converters
//  The three AD inputs are used for the NTCs (temperature measurement)
//
//  Matthias Ulmann, Design Services - EMEA Power Management
//  Texas Instruments, Inc.
//
//******************************************************************************


#include "main.h"



// initialization of analog inputs
// temperature measurements with NTCs, output voltage and current
void init_adc(void)
{
	// select internal reference and GND
	clear_bit(ADC10CTL0, SREF2);
	clear_bit(ADC10CTL0, SREF1);
	set_bit(ADC10CTL0, SREF0);
	
	// set sample-and-hold time to 16 ADC10CLKs
	set_bit(ADC10CTL0, ADC10SHT1);
	clear_bit(ADC10CTL0, ADC10SHT0);
	
	// set the sample rate to 200 ksps
	clear_bit(ADC10CTL0, ADC10SR);
	
	// no reference output
	clear_bit(ADC10CTL0, REFOUT);
	
	// reference always buffered
	clear_bit(ADC10CTL0, REFBURST);
	
	// single sample and conversion
	clear_bit(ADC10CTL0, MSC);
	
	// set reference generator voltage to 2.5V
	set_bit(ADC10CTL0, REF2_5V);
	
	// switch reference generator on
	set_bit(ADC10CTL0, REFON);
	
	// switch ADC on
	set_bit(ADC10CTL0, ADC10ON);
	
	// no interrupt
	clear_bit(ADC10CTL0, ADC10IE);
	
	// select A0 as default
	clear_bit(ADC10CTL1, INCH3);
	clear_bit(ADC10CTL1, INCH2);
	clear_bit(ADC10CTL1, INCH1);
	clear_bit(ADC10CTL1, INCH0);
	
	// ADC10SC bit as sample-and-hold source
	clear_bit(ADC10CTL1, SHS1);
	clear_bit(ADC10CTL1, SHS0);
	
	// ADC10 data as straight binary
	clear_bit(ADC10CTL1, ADC10DF);
	
	// sample-and-hold signal not inverted
	clear_bit(ADC10CTL1, ISSH);
	
	// set clock divider to 8 (2 MHz)
	set_bit(ADC10CTL1, ADC10DIV2);
	set_bit(ADC10CTL1, ADC10DIV1);
	set_bit(ADC10CTL1, ADC10DIV0);
	
	// MCLK as source for ADC
	set_bit(ADC10CTL1, ADC10SSEL1);
	clear_bit(ADC10CTL1, ADC10SSEL0);
	
	// convert a single channel once
	clear_bit(ADC10CTL1, CONSEQ1);
	clear_bit(ADC10CTL1, CONSEQ0);
	
	
	// setup of inputs
	// set pins as inputs
	clear_bit(A0_PDIR, A2_BIT);
	clear_bit(A1_PDIR, A1_BIT);
	clear_bit(A2_PDIR, A0_BIT);
	clear_bit(A6_PDIR, A6_BIT);
	clear_bit(A7_PDIR, A7_BIT);
	
	// disable pull ups
	clear_bit(A0_RES, A2_BIT);
	clear_bit(A1_RES, A1_BIT);
	clear_bit(A2_RES, A0_BIT);
	clear_bit(A6_RES, A6_BIT);
	clear_bit(A7_RES, A7_BIT);
	
	// enable analog input for A0
	clear_bit(ADC10AE0, A7_BIT);
	clear_bit(ADC10AE0, A6_BIT);
	clear_bit(ADC10AE0, A2_BIT);
	clear_bit(ADC10AE0, A1_BIT);
	set_bit(ADC10AE0, A0_BIT);
}



// Conversion of the sampled digital value into degrees.
// The minimum value is 10 degree. For a temperature below a
// value of 0 is returned.
// The maximum value is 100 degree. For a temperature above a
// value of 200 is returned.
unsigned int lsb_to_temperature_adc(unsigned int lsb)
{
	unsigned int degree=0;
	const unsigned int offset=0;	// if needed for calibration
	
	// measurement range is limited by reference voltage
	if(lsb >= 988) degree = 0;
	
	if(lsb < 988 && lsb >= 971) degree = 10 + offset;
	if(lsb < 971 && lsb >= 953) degree = 11 + offset;
	if(lsb < 953 && lsb >= 936) degree = 12 + offset;
	if(lsb < 936 && lsb >= 918) degree = 13 + offset;
	if(lsb < 918 && lsb >= 900) degree = 14 + offset;
	if(lsb < 900 && lsb >= 881) degree = 15 + offset;
	
	if(lsb < 881 && lsb >= 863) degree = 16 + offset;
	if(lsb < 863 && lsb >= 845) degree = 17 + offset;
	if(lsb < 845 && lsb >= 826) degree = 18 + offset;
	if(lsb < 826 && lsb >= 808) degree = 19 + offset;
	if(lsb < 808 && lsb >= 789) degree = 20 + offset;
	
	if(lsb < 789 && lsb >= 770) degree = 21 + offset;
	if(lsb < 770 && lsb >= 752) degree = 22 + offset;
	if(lsb < 752 && lsb >= 733) degree = 23 + offset;
	if(lsb < 733 && lsb >= 715) degree = 24 + offset;
	if(lsb < 715 && lsb >= 697) degree = 25 + offset;
	
	if(lsb < 697 && lsb >= 679) degree = 26 + offset;
	if(lsb < 679 && lsb >= 661) degree = 27 + offset;
	if(lsb < 661 && lsb >= 643) degree = 28 + offset;
	if(lsb < 643 && lsb >= 625) degree = 29 + offset;
	if(lsb < 625 && lsb >= 608) degree = 30 + offset;
	
	if(lsb < 608 && lsb >= 591) degree = 31 + offset;
	if(lsb < 591 && lsb >= 574) degree = 32 + offset;
	if(lsb < 574 && lsb >= 557) degree = 33 + offset;
	if(lsb < 557 && lsb >= 541) degree = 34 + offset;
	if(lsb < 541 && lsb >= 525) degree = 35 + offset;
	
	if(lsb < 525 && lsb >= 509) degree = 36 + offset;
	if(lsb < 509 && lsb >= 493) degree = 37 + offset;
	if(lsb < 493 && lsb >= 478) degree = 38 + offset;
	if(lsb < 478 && lsb >= 463) degree = 39 + offset;
	if(lsb < 463 && lsb >= 449) degree = 40 + offset;
	
	if(lsb < 449 && lsb >= 435) degree = 41 + offset;
	if(lsb < 435 && lsb >= 421) degree = 42 + offset;
	if(lsb < 421 && lsb >= 407) degree = 43 + offset;
	if(lsb < 407 && lsb >= 394) degree = 44 + offset;
	if(lsb < 394 && lsb >= 382) degree = 45 + offset;
	
	if(lsb < 382 && lsb >= 369) degree = 46 + offset;
	if(lsb < 369 && lsb >= 357) degree = 47 + offset;
	if(lsb < 357 && lsb >= 345) degree = 48 + offset;
	if(lsb < 345 && lsb >= 334) degree = 49 + offset;
	if(lsb < 334 && lsb >= 323) degree = 50 + offset;
	
	if(lsb < 323 && lsb >= 312) degree = 51 + offset;
	if(lsb < 312 && lsb >= 301) degree = 52 + offset;
	if(lsb < 301 && lsb >= 291) degree = 53 + offset;
	if(lsb < 291 && lsb >= 281) degree = 54 + offset;
	if(lsb < 281 && lsb >= 272) degree = 55 + offset;
	
	if(lsb < 272 && lsb >= 263) degree = 56 + offset;
	if(lsb < 263 && lsb >= 254) degree = 57 + offset;
	if(lsb < 254 && lsb >= 245) degree = 58 + offset;
	if(lsb < 245 && lsb >= 237) degree = 59 + offset;
	if(lsb < 237 && lsb >= 229) degree = 60 + offset;
	
	if(lsb < 229 && lsb >= 221) degree = 61 + offset;
	if(lsb < 221 && lsb >= 213) degree = 62 + offset;
	if(lsb < 213 && lsb >= 206) degree = 63 + offset;
	if(lsb < 206 && lsb >= 199) degree = 64 + offset;
	if(lsb < 199 && lsb >= 192) degree = 65 + offset;
	
	if(lsb < 192 && lsb >= 186) degree = 66 + offset;
	if(lsb < 186 && lsb >= 179) degree = 67 + offset;
	if(lsb < 179 && lsb >= 173) degree = 68 + offset;
	if(lsb < 173 && lsb >= 167) degree = 69 + offset;
	if(lsb < 167 && lsb >= 162) degree = 70 + offset;
	
	if(lsb < 162 && lsb >= 156) degree = 71 + offset;
	if(lsb < 156 && lsb >= 151) degree = 72 + offset;
	if(lsb < 151 && lsb >= 146) degree = 73 + offset;
	if(lsb < 146 && lsb >= 141) degree = 74 + offset;
	if(lsb < 141 && lsb >= 136) degree = 75 + offset;
	
	if(lsb < 136 && lsb >= 132) degree = 76 + offset;
	if(lsb < 132 && lsb >= 127) degree = 77 + offset;
	if(lsb < 127 && lsb >= 123) degree = 78 + offset;
	if(lsb < 123 && lsb >= 119) degree = 79 + offset;
	if(lsb < 119 && lsb >= 115) degree = 80 + offset;
	
	if(lsb < 115 && lsb >= 111) degree = 81 + offset;
	if(lsb < 111 && lsb >= 107) degree = 82 + offset;
	if(lsb < 107 && lsb >= 104) degree = 83 + offset;
	if(lsb < 104 && lsb >= 100) degree = 84 + offset;
	if(lsb < 100 && lsb >= 97) degree = 85 + offset;
	
	if(lsb < 97 && lsb >= 94) degree = 86 + offset;
	if(lsb < 94 && lsb >= 91) degree = 87 + offset;
	if(lsb < 91 && lsb >= 88) degree = 88 + offset;
	if(lsb < 88 && lsb >= 85) degree = 89 + offset;
	if(lsb < 85 && lsb >= 82) degree = 90 + offset;
	
	if(lsb < 82 && lsb >= 80) degree = 91 + offset;
	if(lsb < 80 && lsb >= 77) degree = 92 + offset;
	if(lsb < 77 && lsb >= 75) degree = 93 + offset;
	if(lsb < 75 && lsb >= 72) degree = 94 + offset;
	if(lsb < 72 && lsb >= 70) degree = 95 + offset;
	
	if(lsb < 70 && lsb >= 68) degree = 96 + offset;
	if(lsb < 68 && lsb >= 66) degree = 97 + offset;
	if(lsb < 66 && lsb >= 64) degree = 98 + offset;
	if(lsb < 64 && lsb >= 62) degree = 99 + offset;
	
	// max. measured temperature is 99 degree
	if (lsb <= 64) degree = 200;
	
	return degree;
}



// Conversion of the sampled digital value into voltage.
// The value has three digits, the last represents hundred millivolts.
// (135 means 13.5V)
unsigned int lsb_to_voltage_adc(unsigned int lsb)
{
	unsigned long voltage_long;
	unsigned int voltage_int;
	
	// The output voltage is attenuated by a voltage divider 5k1/25k5.
	// This formula is used to calculate the output voltage from the LSBs:
	// Vout = LSB * (2.5/1023) * (5k11 + 25k5)/5k11 * 10
	// simplified: Vout = LSB * 0.146
	// To prevent an overflow and the loss of the first decimal place a
	// long integer is needed for calculation.
	
	voltage_long = (unsigned long)lsb * 146;
	voltage_long = voltage_long / 1000;
	
	voltage_int = voltage_long;
	return voltage_int;
}



// Conversion of the sampled digital value into current.
// The value has three digits, the last represents hundred milliamps.
// (135 means 13.5V)
unsigned int lsb_to_current_adc(unsigned int lsb)
{
	unsigned long current_long;
	unsigned int current_int;
	unsigned long offset_long;
	
	// constant value to eliminate an offset error of the measurement
	const unsigned int current_offset = 5;
	
	// 33mV equals 1A
	// This formula is used to calculate the output current from the LSBs:
	// Iout = LSB * 2.5/1023 / 0.033 * 10
	// simplified: Iout = LSB * 0.741 
	// To prevent an overflow and the loss of the first decimal place a
	// long integer is needed for calculation.
	current_long = (unsigned long)lsb * 741;
	current_long = current_long / 1000;
	
	// subtract a constant percentage to eliminate the offset error
	offset_long = current_long * current_offset;
	offset_long = offset_long / 100;
	current_long = current_long - offset_long; 
	
	current_int = current_long;
	return current_int;
}



// Run the AD converter and calculate the temperature of the 3 NTCs
// and the output voltage and current.
void get_analog_inputs(void)
{
	// variables used for reading the analog inputs
	static volatile unsigned int analog_counter=0;
	static volatile unsigned int ntc_hbridge_new=0, ntc_hbridge_old=0;
	static volatile unsigned int ntc_display_new=0, ntc_display_old=0;
	static volatile unsigned int ntc_pfc_new=0, ntc_pfc_old=0;
	static volatile unsigned int analog_voltage_new=0, analog_voltage_old=0;
	static volatile unsigned int analog_current_new=0, analog_current_old=0;
	unsigned int lsb_voltage_output, lsb_current_output;
	unsigned int calculation_lsb;
	
	// Filter constant (Linear Shift-Invariant)
	const unsigned int analog_averaging = 8;
	
	// read ADC results
	// Skip if AD-conversion is not yet finished.
	// This will never happen due to the short conversion time.
	while (ADC10CTL1 & ADC10BUSY);
	
	switch(analog_counter)
	{
		case 0:	ntc_hbridge_old = ntc_hbridge_new;	// save last value
				ntc_hbridge_new = ADC10MEM;			// read ADC memory
				
				// calculate new value and write to global variable in degree
				calculation_lsb =
				ntc_hbridge_old * (analog_averaging - 1)/analog_averaging
				+ (ntc_hbridge_new / analog_averaging);
				
				temp_hbridge = lsb_to_temperature_adc(calculation_lsb);
					
				analog_counter = 1;			// shift channel
				
				// stop ADC to change the input channel
				clear_bit(ADC10CTL0, ENC);
				
				//set channel to A1
				clear_bit(ADC10CTL1, INCH3);
				clear_bit(ADC10CTL1, INCH2);
				clear_bit(ADC10CTL1, INCH1);
				set_bit(ADC10CTL1, INCH0);
					
				// enable analog input for A1
				clear_bit(ADC10AE0, A7_BIT);
				clear_bit(ADC10AE0, A6_BIT);
				clear_bit(ADC10AE0, A2_BIT);
				set_bit(ADC10AE0, A1_BIT);
				clear_bit(ADC10AE0, A0_BIT);
				
				// enable and start conversion
				set_bit(ADC10CTL0, ENC);
   				set_bit(ADC10CTL0, ADC10SC);
   				break;
   					
		case 1:	ntc_display_old = ntc_display_new;	// save last value
				ntc_display_new = ADC10MEM;			// read ADC memory
				
				// calculate new value and write to global variable in degree
				calculation_lsb =
				ntc_display_old * (analog_averaging - 1)/analog_averaging
				+ (ntc_display_new / analog_averaging);
				
				temp_chassis = lsb_to_temperature_adc(calculation_lsb);
				
				analog_counter = 2;			// shift channel
				
				// stop ADC to change the input channel
				clear_bit(ADC10CTL0, ENC);
				
				//set channel to A2
				clear_bit(ADC10CTL1, INCH3);
				clear_bit(ADC10CTL1, INCH2);
				set_bit(ADC10CTL1, INCH1);
				clear_bit(ADC10CTL1, INCH0);
				
				// enable analog input for A2
				clear_bit(ADC10AE0, A7_BIT);
				clear_bit(ADC10AE0, A6_BIT);
				set_bit(ADC10AE0, A2_BIT);
				clear_bit(ADC10AE0, A1_BIT);
				clear_bit(ADC10AE0, A0_BIT);
				
				// enable and start conversion
				set_bit(ADC10CTL0, ENC);
				set_bit(ADC10CTL0, ADC10SC);
				break;
   					
		case 2:	ntc_pfc_old = ntc_pfc_new;	// save last value
				ntc_pfc_new = ADC10MEM;		// read ADC memory
				
				// calculate new value and write to global variable in degree
				calculation_lsb =
				ntc_pfc_old * (analog_averaging - 1)/analog_averaging
				+ (ntc_pfc_new / analog_averaging);
				
				temp_pfc = lsb_to_temperature_adc(calculation_lsb);
				
				analog_counter = 6;			// shift channel
				
				// stop ADC to change the input channel
				clear_bit(ADC10CTL0, ENC);
				
				//set channel to A6
				clear_bit(ADC10CTL1, INCH3);
				set_bit(ADC10CTL1, INCH2);
				set_bit(ADC10CTL1, INCH1);
				clear_bit(ADC10CTL1, INCH0);
				
				// enable analog input for A6
				clear_bit(ADC10AE0, A7_BIT);
				set_bit(ADC10AE0, A6_BIT);
				clear_bit(ADC10AE0, A2_BIT);
				clear_bit(ADC10AE0, A1_BIT);
				clear_bit(ADC10AE0, A0_BIT);
				
				// enable and start conversion
				set_bit(ADC10CTL0, ENC);
				set_bit(ADC10CTL0, ADC10SC);
				break;
				
		case 6:	analog_voltage_old = analog_voltage_new;	// save last value
				analog_voltage_new = ADC10MEM;				// read ADC memory
				
				// calculate new value and write to global variable in volt
				lsb_voltage_output =
				analog_voltage_old * (analog_averaging - 1)/analog_averaging
				+ (analog_voltage_new / analog_averaging);
				
				voltage_output = lsb_to_voltage_adc(lsb_voltage_output);
					
				analog_counter = 7;			// shift channel
				
				// stop ADC to change the input channel
				clear_bit(ADC10CTL0, ENC);
				
				//set channel to A7
				clear_bit(ADC10CTL1, INCH3);
				set_bit(ADC10CTL1, INCH2);
				set_bit(ADC10CTL1, INCH1);
				set_bit(ADC10CTL1, INCH0);
					
				// enable analog input for A7
				set_bit(ADC10AE0, A7_BIT);
				clear_bit(ADC10AE0, A6_BIT);
				clear_bit(ADC10AE0, A2_BIT);
				clear_bit(ADC10AE0, A1_BIT);
				clear_bit(ADC10AE0, A0_BIT);
				
				// enable and start conversion
				set_bit(ADC10CTL0, ENC);
   				set_bit(ADC10CTL0, ADC10SC);
   				break;
   					
		case 7:	analog_current_old = analog_current_new;	// save last value
				analog_current_new = ADC10MEM;				// read ADC memory
				
				// calculate new value and write to global variable in ampere
				lsb_current_output =
				analog_current_old * (analog_averaging - 1)/analog_averaging
				+ (analog_current_new / analog_averaging);
				
				current_output = lsb_to_current_adc(lsb_current_output);
				
				analog_counter = 0;			// shift channel
				
				// stop ADC to change the input channel
				clear_bit(ADC10CTL0, ENC);
				
				//set channel to A0
				clear_bit(ADC10CTL1, INCH3);
				clear_bit(ADC10CTL1, INCH2);
				clear_bit(ADC10CTL1, INCH1);
				clear_bit(ADC10CTL1, INCH0);
				
				// enable analog input for A0
				clear_bit(ADC10AE0, A7_BIT);
				clear_bit(ADC10AE0, A6_BIT);
				clear_bit(ADC10AE0, A2_BIT);
				clear_bit(ADC10AE0, A1_BIT);
				set_bit(ADC10AE0, A0_BIT);
				
				// enable and start conversion
				set_bit(ADC10CTL0, ENC);
				set_bit(ADC10CTL0, ADC10SC);
				break;
			
	default:	break;
	}	
}
